use lazy_static::lazy_static;
use std::collections::HashMap;

#[derive(Copy, Clone)]
pub(crate) struct UnicodeBestfitMap {
    // Best-fit mapping options.
    /// The best-fit map to use to decode %u-encoded characters.
    pub(crate) map: &'static HashMap<u32, u8>,
    /// The replacement byte used when there is no best-fit mapping.
    pub(crate) replacement_byte: u8,
}

impl Default for UnicodeBestfitMap {
    fn default() -> Self {
        Self {
            map: &bestfit_1252,
            replacement_byte: b'?',
        }
    }
}

impl UnicodeBestfitMap {
    pub(crate) fn get(&self, unicode: u32) -> u8 {
        self.map
            .get(&unicode)
            .copied()
            .unwrap_or(self.replacement_byte)
    }
}

/// Create bestfit key from two 8 bit bytes.
#[macro_export]
macro_rules! bestfit_key {
    ($c1:expr, $c2:expr) => {
        ((($c1 as i32) << 8 as i32) + $c2 as i32) as u32
    };
}

lazy_static! {
    static ref bestfit_1252: HashMap<u32, u8> = [
        (bestfit_key!(0x01, 0), 0x41),
        (bestfit_key!(0x1, 0x1), 0x61),
        (bestfit_key!(0x1, 0x2), 0x41),
        (bestfit_key!(0x1, 0x3), 0x61),
        (bestfit_key!(0x1, 0x4), 0x41),
        (bestfit_key!(0x1, 0x5), 0x61),
        (bestfit_key!(0x1, 0x6), 0x43),
        (bestfit_key!(0x1, 0x7), 0x63),
        (bestfit_key!(0x1, 0x8), 0x43),
        (bestfit_key!(0x1, 0x9), 0x63),
        (bestfit_key!(0x1, 0xa), 0x43),
        (bestfit_key!(0x1, 0xb), 0x63),
        (bestfit_key!(0x1, 0xc), 0x43),
        (bestfit_key!(0x1, 0xd), 0x63),
        (bestfit_key!(0x1, 0xe), 0x44),
        (bestfit_key!(0x1, 0xf), 0x64),
        (bestfit_key!(0x1, 0x11), 0x64),
        (bestfit_key!(0x1, 0x12), 0x45),
        (bestfit_key!(0x1, 0x13), 0x65),
        (bestfit_key!(0x1, 0x14), 0x45),
        (bestfit_key!(0x1, 0x15), 0x65),
        (bestfit_key!(0x1, 0x16), 0x45),
        (bestfit_key!(0x1, 0x17), 0x65),
        (bestfit_key!(0x1, 0x18), 0x45),
        (bestfit_key!(0x1, 0x19), 0x65),
        (bestfit_key!(0x1, 0x1a), 0x45),
        (bestfit_key!(0x1, 0x1b), 0x65),
        (bestfit_key!(0x1, 0x1c), 0x47),
        (bestfit_key!(0x1, 0x1d), 0x67),
        (bestfit_key!(0x1, 0x1e), 0x47),
        (bestfit_key!(0x1, 0x1f), 0x67),
        (bestfit_key!(0x1, 0x20), 0x47),
        (bestfit_key!(0x1, 0x21), 0x67),
        (bestfit_key!(0x1, 0x22), 0x47),
        (bestfit_key!(0x1, 0x23), 0x67),
        (bestfit_key!(0x1, 0x24), 0x48),
        (bestfit_key!(0x1, 0x25), 0x68),
        (bestfit_key!(0x1, 0x26), 0x48),
        (bestfit_key!(0x1, 0x27), 0x68),
        (bestfit_key!(0x1, 0x28), 0x49),
        (bestfit_key!(0x1, 0x29), 0x69),
        (bestfit_key!(0x1, 0x2a), 0x49),
        (bestfit_key!(0x1, 0x2b), 0x69),
        (bestfit_key!(0x1, 0x2c), 0x49),
        (bestfit_key!(0x1, 0x2d), 0x69),
        (bestfit_key!(0x1, 0x2e), 0x49),
        (bestfit_key!(0x1, 0x2f), 0x69),
        (bestfit_key!(0x1, 0x30), 0x49),
        (bestfit_key!(0x1, 0x31), 0x69),
        (bestfit_key!(0x1, 0x34), 0x4a),
        (bestfit_key!(0x1, 0x35), 0x6a),
        (bestfit_key!(0x1, 0x36), 0x4b),
        (bestfit_key!(0x1, 0x37), 0x6b),
        (bestfit_key!(0x1, 0x39), 0x4c),
        (bestfit_key!(0x1, 0x3a), 0x6c),
        (bestfit_key!(0x1, 0x3b), 0x4c),
        (bestfit_key!(0x1, 0x3c), 0x6c),
        (bestfit_key!(0x1, 0x3d), 0x4c),
        (bestfit_key!(0x1, 0x3e), 0x6c),
        (bestfit_key!(0x1, 0x41), 0x4c),
        (bestfit_key!(0x1, 0x42), 0x6c),
        (bestfit_key!(0x1, 0x43), 0x4e),
        (bestfit_key!(0x1, 0x44), 0x6e),
        (bestfit_key!(0x1, 0x45), 0x4e),
        (bestfit_key!(0x1, 0x46), 0x6e),
        (bestfit_key!(0x1, 0x47), 0x4e),
        (bestfit_key!(0x1, 0x48), 0x6e),
        (bestfit_key!(0x1, 0x4c), 0x4f),
        (bestfit_key!(0x1, 0x4d), 0x6f),
        (bestfit_key!(0x1, 0x4e), 0x4f),
        (bestfit_key!(0x1, 0x4f), 0x6f),
        (bestfit_key!(0x1, 0x50), 0x4f),
        (bestfit_key!(0x1, 0x51), 0x6f),
        (bestfit_key!(0x1, 0x54), 0x52),
        (bestfit_key!(0x1, 0x55), 0x72),
        (bestfit_key!(0x1, 0x56), 0x52),
        (bestfit_key!(0x1, 0x57), 0x72),
        (bestfit_key!(0x1, 0x58), 0x52),
        (bestfit_key!(0x1, 0x59), 0x72),
        (bestfit_key!(0x1, 0x5a), 0x53),
        (bestfit_key!(0x1, 0x5b), 0x73),
        (bestfit_key!(0x1, 0x5c), 0x53),
        (bestfit_key!(0x1, 0x5d), 0x73),
        (bestfit_key!(0x1, 0x5e), 0x53),
        (bestfit_key!(0x1, 0x5f), 0x73),
        (bestfit_key!(0x1, 0x62), 0x54),
        (bestfit_key!(0x1, 0x63), 0x74),
        (bestfit_key!(0x1, 0x64), 0x54),
        (bestfit_key!(0x1, 0x65), 0x74),
        (bestfit_key!(0x1, 0x66), 0x54),
        (bestfit_key!(0x1, 0x67), 0x74),
        (bestfit_key!(0x1, 0x68), 0x55),
        (bestfit_key!(0x1, 0x69), 0x75),
        (bestfit_key!(0x1, 0x6a), 0x55),
        (bestfit_key!(0x1, 0x6b), 0x75),
        (bestfit_key!(0x1, 0x6c), 0x55),
        (bestfit_key!(0x1, 0x6d), 0x75),
        (bestfit_key!(0x1, 0x6e), 0x55),
        (bestfit_key!(0x1, 0x6f), 0x75),
        (bestfit_key!(0x1, 0x70), 0x55),
        (bestfit_key!(0x1, 0x71), 0x75),
        (bestfit_key!(0x1, 0x72), 0x55),
        (bestfit_key!(0x1, 0x73), 0x75),
        (bestfit_key!(0x1, 0x74), 0x57),
        (bestfit_key!(0x1, 0x75), 0x77),
        (bestfit_key!(0x1, 0x76), 0x59),
        (bestfit_key!(0x1, 0x77), 0x79),
        (bestfit_key!(0x1, 0x79), 0x5a),
        (bestfit_key!(0x1, 0x7b), 0x5a),
        (bestfit_key!(0x1, 0x7c), 0x7a),
        (bestfit_key!(0x1, 0x80), 0x62),
        (bestfit_key!(0x1, 0x97), 0x49),
        (bestfit_key!(0x1, 0x9a), 0x6c),
        (bestfit_key!(0x1, 0x9f), 0x4f),
        (bestfit_key!(0x1, 0xa0), 0x4f),
        (bestfit_key!(0x1, 0xa1), 0x6f),
        (bestfit_key!(0x1, 0xab), 0x74),
        (bestfit_key!(0x1, 0xae), 0x54),
        (bestfit_key!(0x1, 0xaf), 0x55),
        (bestfit_key!(0x1, 0xb0), 0x75),
        (bestfit_key!(0x1, 0xb6), 0x7a),
        (bestfit_key!(0x1, 0xc0), 0x7c),
        (bestfit_key!(0x1, 0xc3), 0x21),
        (bestfit_key!(0x1, 0xcd), 0x41),
        (bestfit_key!(0x1, 0xce), 0x61),
        (bestfit_key!(0x1, 0xcf), 0x49),
        (bestfit_key!(0x1, 0xd0), 0x69),
        (bestfit_key!(0x1, 0xd1), 0x4f),
        (bestfit_key!(0x1, 0xd2), 0x6f),
        (bestfit_key!(0x1, 0xd3), 0x55),
        (bestfit_key!(0x1, 0xd4), 0x75),
        (bestfit_key!(0x1, 0xd5), 0x55),
        (bestfit_key!(0x1, 0xd6), 0x75),
        (bestfit_key!(0x1, 0xd7), 0x55),
        (bestfit_key!(0x1, 0xd8), 0x75),
        (bestfit_key!(0x1, 0xd9), 0x55),
        (bestfit_key!(0x1, 0xda), 0x75),
        (bestfit_key!(0x1, 0xdb), 0x55),
        (bestfit_key!(0x1, 0xdc), 0x75),
        (bestfit_key!(0x1, 0xde), 0x41),
        (bestfit_key!(0x1, 0xdf), 0x61),
        (bestfit_key!(0x1, 0xe4), 0x47),
        (bestfit_key!(0x1, 0xe5), 0x67),
        (bestfit_key!(0x1, 0xe6), 0x47),
        (bestfit_key!(0x1, 0xe7), 0x67),
        (bestfit_key!(0x1, 0xe8), 0x4b),
        (bestfit_key!(0x1, 0xe9), 0x6b),
        (bestfit_key!(0x1, 0xea), 0x4f),
        (bestfit_key!(0x1, 0xeb), 0x6f),
        (bestfit_key!(0x1, 0xec), 0x4f),
        (bestfit_key!(0x1, 0xed), 0x6f),
        (bestfit_key!(0x1, 0xf0), 0x6a),
        (bestfit_key!(0x2, 0x61), 0x67),
        (bestfit_key!(0x2, 0xb9), 0x27),
        (bestfit_key!(0x2, 0xba), 0x22),
        (bestfit_key!(0x2, 0xbc), 0x27),
        (bestfit_key!(0x2, 0xc4), 0x5e),
        (bestfit_key!(0x2, 0xc8), 0x27),
        (bestfit_key!(0x2, 0xcb), 0x60),
        (bestfit_key!(0x2, 0xcd), 0x5f),
        (bestfit_key!(0x3, 0x00), 0x60),
        (bestfit_key!(0x3, 0x2), 0x5e),
        (bestfit_key!(0x3, 0x3), 0x7e),
        (bestfit_key!(0x3, 0xe), 0x22),
        (bestfit_key!(0x3, 0x31), 0x5f),
        (bestfit_key!(0x3, 0x32), 0x5f),
        (bestfit_key!(0x3, 0x7e), 0x3b),
        (bestfit_key!(0x3, 0x93), 0x47),
        (bestfit_key!(0x3, 0x98), 0x54),
        (bestfit_key!(0x3, 0xa3), 0x53),
        (bestfit_key!(0x3, 0xa6), 0x46),
        (bestfit_key!(0x3, 0xa9), 0x4f),
        (bestfit_key!(0x3, 0xb1), 0x61),
        (bestfit_key!(0x3, 0xb4), 0x64),
        (bestfit_key!(0x3, 0xb5), 0x65),
        (bestfit_key!(0x3, 0xc0), 0x70),
        (bestfit_key!(0x3, 0xc3), 0x73),
        (bestfit_key!(0x3, 0xc4), 0x74),
        (bestfit_key!(0x3, 0xc6), 0x66),
        (bestfit_key!(0x4, 0xbb), 0x68),
        (bestfit_key!(0x5, 0x89), 0x3a),
        (bestfit_key!(0x6, 0x6a), 0x25),
        (bestfit_key!(0x20, 0), 0x20),
        (bestfit_key!(0x20, 0x1), 0x20),
        (bestfit_key!(0x20, 0x2), 0x20),
        (bestfit_key!(0x20, 0x3), 0x20),
        (bestfit_key!(0x20, 0x4), 0x20),
        (bestfit_key!(0x20, 0x5), 0x20),
        (bestfit_key!(0x20, 0x6), 0x20),
        (bestfit_key!(0x20, 0x10), 0x2d),
        (bestfit_key!(0x20, 0x11), 0x2d),
        (bestfit_key!(0x20, 0x17), 0x3d),
        (bestfit_key!(0x20, 0x32), 0x27),
        (bestfit_key!(0x20, 0x35), 0x60),
        (bestfit_key!(0x20, 0x44), 0x2f),
        (bestfit_key!(0x20, 0x74), 0x34),
        (bestfit_key!(0x20, 0x75), 0x35),
        (bestfit_key!(0x20, 0x76), 0x36),
        (bestfit_key!(0x20, 0x77), 0x37),
        (bestfit_key!(0x20, 0x78), 0x38),
        (bestfit_key!(0x20, 0x7f), 0x6e),
        (bestfit_key!(0x20, 0x80), 0x30),
        (bestfit_key!(0x20, 0x81), 0x31),
        (bestfit_key!(0x20, 0x82), 0x32),
        (bestfit_key!(0x20, 0x83), 0x33),
        (bestfit_key!(0x20, 0x84), 0x34),
        (bestfit_key!(0x20, 0x85), 0x35),
        (bestfit_key!(0x20, 0x86), 0x36),
        (bestfit_key!(0x20, 0x87), 0x37),
        (bestfit_key!(0x20, 0x88), 0x38),
        (bestfit_key!(0x20, 0x89), 0x39),
        (bestfit_key!(0x20, 0xa7), 0x50),
        (bestfit_key!(0x21, 0x2), 0x43),
        (bestfit_key!(0x21, 0x7), 0x45),
        (bestfit_key!(0x21, 0xa), 0x67),
        (bestfit_key!(0x21, 0xb), 0x48),
        (bestfit_key!(0x21, 0xc), 0x48),
        (bestfit_key!(0x21, 0xd), 0x48),
        (bestfit_key!(0x21, 0xe), 0x68),
        (bestfit_key!(0x21, 0x10), 0x49),
        (bestfit_key!(0x21, 0x11), 0x49),
        (bestfit_key!(0x21, 0x12), 0x4c),
        (bestfit_key!(0x21, 0x13), 0x6c),
        (bestfit_key!(0x21, 0x15), 0x4e),
        (bestfit_key!(0x21, 0x18), 0x50),
        (bestfit_key!(0x21, 0x19), 0x50),
        (bestfit_key!(0x21, 0x1a), 0x51),
        (bestfit_key!(0x21, 0x1b), 0x52),
        (bestfit_key!(0x21, 0x1c), 0x52),
        (bestfit_key!(0x21, 0x1d), 0x52),
        (bestfit_key!(0x21, 0x24), 0x5a),
        (bestfit_key!(0x21, 0x28), 0x5a),
        (bestfit_key!(0x21, 0x2a), 0x4b),
        (bestfit_key!(0x21, 0x2c), 0x42),
        (bestfit_key!(0x21, 0x2d), 0x43),
        (bestfit_key!(0x21, 0x2e), 0x65),
        (bestfit_key!(0x21, 0x2f), 0x65),
        (bestfit_key!(0x21, 0x30), 0x45),
        (bestfit_key!(0x21, 0x31), 0x46),
        (bestfit_key!(0x21, 0x33), 0x4d),
        (bestfit_key!(0x21, 0x34), 0x6f),
        (bestfit_key!(0x22, 0x12), 0x2d),
        (bestfit_key!(0x22, 0x15), 0x2f),
        (bestfit_key!(0x22, 0x16), 0x5c),
        (bestfit_key!(0x22, 0x17), 0x2a),
        (bestfit_key!(0x22, 0x1a), 0x76),
        (bestfit_key!(0x22, 0x1e), 0x38),
        (bestfit_key!(0x22, 0x23), 0x7c),
        (bestfit_key!(0x22, 0x29), 0x6e),
        (bestfit_key!(0x22, 0x36), 0x3a),
        (bestfit_key!(0x22, 0x3c), 0x7e),
        (bestfit_key!(0x22, 0x61), 0x3d),
        (bestfit_key!(0x22, 0x64), 0x3d),
        (bestfit_key!(0x22, 0x65), 0x3d),
        (bestfit_key!(0x23, 0x3), 0x5e),
        (bestfit_key!(0x23, 0x20), 0x28),
        (bestfit_key!(0x23, 0x21), 0x29),
        (bestfit_key!(0x23, 0x29), 0x3c),
        (bestfit_key!(0x23, 0x2a), 0x3e),
        (bestfit_key!(0x25, 0), 0x2d),
        (bestfit_key!(0x25, 0xc), 0x2b),
        (bestfit_key!(0x25, 0x10), 0x2b),
        (bestfit_key!(0x25, 0x14), 0x2b),
        (bestfit_key!(0x25, 0x18), 0x2b),
        (bestfit_key!(0x25, 0x1c), 0x2b),
        (bestfit_key!(0x25, 0x2c), 0x2d),
        (bestfit_key!(0x25, 0x34), 0x2d),
        (bestfit_key!(0x25, 0x3c), 0x2b),
        (bestfit_key!(0x25, 0x50), 0x2d),
        (bestfit_key!(0x25, 0x52), 0x2b),
        (bestfit_key!(0x25, 0x53), 0x2b),
        (bestfit_key!(0x25, 0x54), 0x2b),
        (bestfit_key!(0x25, 0x55), 0x2b),
        (bestfit_key!(0x25, 0x56), 0x2b),
        (bestfit_key!(0x25, 0x57), 0x2b),
        (bestfit_key!(0x25, 0x58), 0x2b),
        (bestfit_key!(0x25, 0x59), 0x2b),
        (bestfit_key!(0x25, 0x5a), 0x2b),
        (bestfit_key!(0x25, 0x5b), 0x2b),
        (bestfit_key!(0x25, 0x5c), 0x2b),
        (bestfit_key!(0x25, 0x5d), 0x2b),
        (bestfit_key!(0x25, 0x64), 0x2d),
        (bestfit_key!(0x25, 0x65), 0x2d),
        (bestfit_key!(0x25, 0x66), 0x2d),
        (bestfit_key!(0x25, 0x67), 0x2d),
        (bestfit_key!(0x25, 0x68), 0x2d),
        (bestfit_key!(0x25, 0x69), 0x2d),
        (bestfit_key!(0x25, 0x6a), 0x2b),
        (bestfit_key!(0x25, 0x6b), 0x2b),
        (bestfit_key!(0x25, 0x6c), 0x2b),
        (bestfit_key!(0x25, 0x84), 0x5f),
        (bestfit_key!(0x27, 0x58), 0x7c),
        (bestfit_key!(0x30, 0), 0x20),
        (bestfit_key!(0x30, 0x8), 0x3c),
        (bestfit_key!(0x30, 0x9), 0x3e),
        (bestfit_key!(0x30, 0x1a), 0x5b),
        (bestfit_key!(0x30, 0x1b), 0x5d),
        (bestfit_key!(0xff, 0x1), 0x21),
        (bestfit_key!(0xff, 0x2), 0x22),
        (bestfit_key!(0xff, 0x3), 0x23),
        (bestfit_key!(0xff, 0x4), 0x24),
        (bestfit_key!(0xff, 0x5), 0x25),
        (bestfit_key!(0xff, 0x6), 0x26),
        (bestfit_key!(0xff, 0x7), 0x27),
        (bestfit_key!(0xff, 0x8), 0x28),
        (bestfit_key!(0xff, 0x9), 0x29),
        (bestfit_key!(0xff, 0xa), 0x2a),
        (bestfit_key!(0xff, 0xb), 0x2b),
        (bestfit_key!(0xff, 0xc), 0x2c),
        (bestfit_key!(0xff, 0xd), 0x2d),
        (bestfit_key!(0xff, 0xe), 0x2e),
        (bestfit_key!(0xff, 0xf), 0x2f),
        (bestfit_key!(0xff, 0x10), 0x30),
        (bestfit_key!(0xff, 0x11), 0x31),
        (bestfit_key!(0xff, 0x12), 0x32),
        (bestfit_key!(0xff, 0x13), 0x33),
        (bestfit_key!(0xff, 0x14), 0x34),
        (bestfit_key!(0xff, 0x15), 0x35),
        (bestfit_key!(0xff, 0x16), 0x36),
        (bestfit_key!(0xff, 0x17), 0x37),
        (bestfit_key!(0xff, 0x18), 0x38),
        (bestfit_key!(0xff, 0x19), 0x39),
        (bestfit_key!(0xff, 0x1a), 0x3a),
        (bestfit_key!(0xff, 0x1b), 0x3b),
        (bestfit_key!(0xff, 0x1c), 0x3c),
        (bestfit_key!(0xff, 0x1d), 0x3d),
        (bestfit_key!(0xff, 0x1e), 0x3e),
        (bestfit_key!(0xff, 0x20), 0x40),
        (bestfit_key!(0xff, 0x21), 0x41),
        (bestfit_key!(0xff, 0x22), 0x42),
        (bestfit_key!(0xff, 0x23), 0x43),
        (bestfit_key!(0xff, 0x24), 0x44),
        (bestfit_key!(0xff, 0x25), 0x45),
        (bestfit_key!(0xff, 0x26), 0x46),
        (bestfit_key!(0xff, 0x27), 0x47),
        (bestfit_key!(0xff, 0x28), 0x48),
        (bestfit_key!(0xff, 0x29), 0x49),
        (bestfit_key!(0xff, 0x2a), 0x4a),
        (bestfit_key!(0xff, 0x2b), 0x4b),
        (bestfit_key!(0xff, 0x2c), 0x4c),
        (bestfit_key!(0xff, 0x2d), 0x4d),
        (bestfit_key!(0xff, 0x2e), 0x4e),
        (bestfit_key!(0xff, 0x2f), 0x4f),
        (bestfit_key!(0xff, 0x30), 0x50),
        (bestfit_key!(0xff, 0x31), 0x51),
        (bestfit_key!(0xff, 0x32), 0x52),
        (bestfit_key!(0xff, 0x33), 0x53),
        (bestfit_key!(0xff, 0x34), 0x54),
        (bestfit_key!(0xff, 0x35), 0x55),
        (bestfit_key!(0xff, 0x36), 0x56),
        (bestfit_key!(0xff, 0x37), 0x57),
        (bestfit_key!(0xff, 0x38), 0x58),
        (bestfit_key!(0xff, 0x39), 0x59),
        (bestfit_key!(0xff, 0x3a), 0x5a),
        (bestfit_key!(0xff, 0x3b), 0x5b),
        (bestfit_key!(0xff, 0x3c), 0x5c),
        (bestfit_key!(0xff, 0x3d), 0x5d),
        (bestfit_key!(0xff, 0x3e), 0x5e),
        (bestfit_key!(0xff, 0x3f), 0x5f),
        (bestfit_key!(0xff, 0x40), 0x60),
        (bestfit_key!(0xff, 0x41), 0x61),
        (bestfit_key!(0xff, 0x42), 0x62),
        (bestfit_key!(0xff, 0x43), 0x63),
        (bestfit_key!(0xff, 0x44), 0x64),
        (bestfit_key!(0xff, 0x45), 0x65),
        (bestfit_key!(0xff, 0x46), 0x66),
        (bestfit_key!(0xff, 0x47), 0x67),
        (bestfit_key!(0xff, 0x48), 0x68),
        (bestfit_key!(0xff, 0x49), 0x69),
        (bestfit_key!(0xff, 0x4a), 0x6a),
        (bestfit_key!(0xff, 0x4b), 0x6b),
        (bestfit_key!(0xff, 0x4c), 0x6c),
        (bestfit_key!(0xff, 0x4d), 0x6d),
        (bestfit_key!(0xff, 0x4e), 0x6e),
        (bestfit_key!(0xff, 0x4f), 0x6f),
        (bestfit_key!(0xff, 0x50), 0x70),
        (bestfit_key!(0xff, 0x51), 0x71),
        (bestfit_key!(0xff, 0x52), 0x72),
        (bestfit_key!(0xff, 0x53), 0x73),
        (bestfit_key!(0xff, 0x54), 0x74),
        (bestfit_key!(0xff, 0x55), 0x75),
        (bestfit_key!(0xff, 0x56), 0x76),
        (bestfit_key!(0xff, 0x57), 0x77),
        (bestfit_key!(0xff, 0x58), 0x78),
        (bestfit_key!(0xff, 0x59), 0x79),
        (bestfit_key!(0xff, 0x5a), 0x7a),
        (bestfit_key!(0xff, 0x5b), 0x7b),
        (bestfit_key!(0xff, 0x5c), 0x7c),
        (bestfit_key!(0xff, 0x5d), 0x7d),
        (bestfit_key!(0xff, 0x5e), 0x7e),
    ]
    .iter()
    .cloned()
    .collect();
}
