#!/usr/bin/env bash

# Peut être optionnellement appelé avec le nom d'un répertoire en argument
# (un chemin partiel, pas forcément directement à la racine de source/)
# Exemples :
## 1_generalites
## 1_generalites/glossaire
## glossaire

FAQ_SRC_BASE_DIR=source
OUTPUT_PREFIX=patch-espaces

information() {
    echo '
Ce script remplace les espaces simples par des espaces insécables
fines ou normales là où l’exigent les normes typographiques
françaises. En principe, il ne devrait pas opérer dans les blocs
de code, et sur les espaces qui précèdent des signes de
ponctuation utilisés comme caractères de balisage en Markdown.
Cependant, une vérification est toujours nécessaire avant
d’appliquer les modifications.

À la fin du processus, toutes les modifications effectuées par le
présent script sur les fichiers `.md` vont être annulées. À la
place, vous obtiendrez un patch que vous pourrez ensuite inspecter
et corriger avant de l’appliquer à votre copie de la FAQ et
d’enregistrer les modifications.

Il est possible de restreindre l‘application de ce script à un
sous-répertoire du répertoire source/ de la FAQ. Il n’est pas
nécessaire d’indiquer le chemin entier du sous-répertoire en
question du moment que la partie que vous renseignez est unique.
Exemples (équivalents) :
  ./espaces-insecables-fr.sh glossaire
  ./espaces-insecables-fr.sh 1_generalites/glossaire
'
    sleep 1
   if [[ "$(grep -F 'Changes not staged for commit:' <<< "$(git status)")" ]] ; then
       echo 'ATTENTION : Votre copie de la FAQ contient des modifications
que vous n’avez pas encore enregistrées avec `git commit`.
Veuillez les valider (`git commit`) ou les annuler (`git stash`)
avant de relancer le script. Voici la liste des fichiers concernés :
'
       sleep 1
       echo "$(git status --untracked-files=no |
                          sed '/^no changes added to commit/d' |
                          sed 's/^/  /')"
   exit 1
   fi
}

make_patch() {
    git diff --patch --output="$1" && git stash
}

begin_of_protected_block() {
    line="$1"
    i="$2"
    if [[ "$(grep -E \
                  -e '^[[:space:]]*~~~' \
                  -e '^[[:space:]]*```' \
                  <<< "$line")" ||
              ( $i == 1 && "${line:0:3}" == '---' ) ]] ; then
        echo true
    fi
}

set_non_breakable_spaces() {
    file="$1"
    i="$2"
    if [[ "$(sed -n "$i"p "$file")" ]] ; then
        # Colons are unlikely to happen at the beginning of a line
        # unless as markup in a definition list
        # (punctuation marks are generally rejected to the next line
        # because they are the last character in a link text).
        sed -i -E "${i}N;$(($i+1))"'s/\n[[:space:]]*([;!?])/ \1/' "$file"
        sed -i -E "${i}N;$(($i+1))"'s/\n[[:space:]]*»/ »/' "$file"
        sed -i -E "${i}N;$(($i+1))"'s/«\n[[:space:]]*/« /' "$file"
        sed -i -E "$i"'s/([^|]) ([;!?])/\1 \2/g' "$file"
        sed -i -E "$i"'s/([^|[:space:]]) ([:»])/\1 \2/g' "$file"
        sed -i -E "$i"'s/« /« /g' "$file"
    fi
}

######## main ########

information

sleep 1
echo 'Travail en cours, veuillez patienter…'
echo ''

while read -r file 0<&3 ; do
    i=0
    state=normal
    special_block_closing=''
    while read -r line 0<&4 ; do
        i=$(($i+1))
        if [[ "$state" == 'normal' ]] ; then
            if [[ "$(begin_of_protected_block "$line" "$i")" ]] ; then
                state=protected
                special_block_closing="$(sed -E 's/^([[:space:]]*...).*$/\1/' <<< "$line")"
            else
                set_non_breakable_spaces "$file" "$i"
            fi
        else
            if [[ "$(grep -E '^'"$special_block_closing" <<< "$line")" ]] ; then
                state=normal
                special_block_closing=''
            fi
        fi
    done 4< "$file"
done 3<<< $(find "$FAQ_SRC_BASE_DIR" -path '*'"$1"'*' -name '*.md' -type f -not -path "$FAQ_SRC_BASE_DIR"'/8_contribuer/*')
exec 3<&-
exec 4<&-

OUTPUT_FILE="${OUTPUT_PREFIX}_$(date '+%y-%m-%d_%H-%M').diff"

make_patch "$OUTPUT_FILE"
echo 'Veuillez vérifier et éventuellement corriger '"$OUTPUT_FILE"
echo 'avant de l’appliquer avec `git apply '"$OUTPUT_FILE"'`.'
