/* Copyright (c) 2001-2004, Roger Dingledine.
 * Copyright (c) 2004-2006, Roger Dingledine, Nick Mathewson.
 * Copyright (c) 2007-2019, The Tor Project, Inc. */
/* See LICENSE for licensing information */

#include "orconfig.h"

#define CRYPTO_OPE_PRIVATE

#include "lib/cc/compat_compiler.h"
#include "lib/crypt_ops/crypto_ope.h"
#include "lib/crypt_ops/crypto_cipher.h"
#include "lib/encoding/binascii.h"
#include "lib/malloc/malloc.h"
#include "test/test.h"
#include "tinytest.h"

#include <stddef.h>
#include <string.h>

static void
test_crypto_ope_consistency(void *arg)
{
  (void)arg;

  crypto_ope_t *ope = NULL;
  crypto_cipher_t *aes = NULL;
  const int TEST_VALS[] = { 5, 500, 1023, 1024, 1025, 2046, 2047, 2048, 2049,
                            10000, OPE_INPUT_MAX };
  unsigned i;
  const uint8_t key[32] = "A fixed key, chosen arbitrarily.";

  ope = crypto_ope_new(key);
  tt_assert(ope);

  uint64_t last_val = 0;
  for (i = 0; i < ARRAY_LENGTH(TEST_VALS); ++i) {
    aes = ope_get_cipher(ope, 0);
    int val = TEST_VALS[i];
    uint64_t v1 = crypto_ope_encrypt(ope, val);
    uint64_t v2 = sum_values_from_cipher(aes, val);
    tt_u64_op(v1, OP_EQ, v2);
    tt_u64_op(v2, OP_GT, last_val);
    last_val = v2;
    crypto_cipher_free(aes);
  }

 done:
  crypto_cipher_free(aes);
  crypto_ope_free(ope);
}

static void
test_crypto_ope_oob(void *arg)
{
  (void)arg;

  crypto_ope_t *ope = NULL;
  const uint8_t key[32] = "A fixed key, chosen arbitrarily.";
  ope = crypto_ope_new(key);

  tt_u64_op(UINT64_MAX, OP_EQ, crypto_ope_encrypt(ope,INT_MIN));
  tt_u64_op(UINT64_MAX, OP_EQ, crypto_ope_encrypt(ope,-100));
  tt_u64_op(UINT64_MAX, OP_EQ, crypto_ope_encrypt(ope,0));
  tt_u64_op(UINT64_MAX, OP_NE, crypto_ope_encrypt(ope,1));
  tt_u64_op(UINT64_MAX, OP_NE, crypto_ope_encrypt(ope,7000));
  tt_u64_op(UINT64_MAX, OP_NE, crypto_ope_encrypt(ope,OPE_INPUT_MAX));
  tt_u64_op(UINT64_MAX, OP_EQ, crypto_ope_encrypt(ope,OPE_INPUT_MAX+1));
  tt_u64_op(UINT64_MAX, OP_EQ, crypto_ope_encrypt(ope,INT_MAX));
 done:
  crypto_ope_free(ope);
}

static const char OPE_TEST_KEY[] =
  "19e05891d55232c08c2cad91d612fdb9cbd6691949a0742434a76c80bc6992fe";

/* generated by a separate python implementation. */
static const struct {
  int v;
  uint64_t r;
} OPE_TEST_VECTORS[] = {
  { 121132, UINT64_C(3971694514) },
  { 82283, UINT64_C(2695743564) },
  { 72661, UINT64_C(2381548866) },
  { 72941, UINT64_C(2390408421) },
  { 123122, UINT64_C(4036781069) },
  { 12154, UINT64_C(402067100) },
  { 121574, UINT64_C(3986197593) },
  { 11391, UINT64_C(376696838) },
  { 65845, UINT64_C(2161801517) },
  { 86301, UINT64_C(2828270975) },
  { 61284, UINT64_C(2013616892) },
  { 70505, UINT64_C(2313368870) },
  { 30438, UINT64_C(1001394664) },
  { 60150, UINT64_C(1977329668) },
  { 114800, UINT64_C(3764946628) },
  { 109403, UINT64_C(3585352477) },
  { 21893, UINT64_C(721388468) },
  { 123569, UINT64_C(4051780471) },
  { 95617, UINT64_C(3134921876) },
  { 48561, UINT64_C(1597596985) },
  { 53334, UINT64_C(1753691710) },
  { 92746, UINT64_C(3040874493) },
  { 7110, UINT64_C(234966492) },
  { 9612, UINT64_C(318326551) },
  { 106958, UINT64_C(3506124249) },
  { 46889, UINT64_C(1542219146) },
  { 87790, UINT64_C(2877361609) },
  { 68878, UINT64_C(2260369112) },
  { 47917, UINT64_C(1576681737) },
  { 121128, UINT64_C(3971553290) },
  { 108602, UINT64_C(3559176081) },
  { 28217, UINT64_C(929692460) },
  { 69498, UINT64_C(2280554161) },
  { 63870, UINT64_C(2098322675) },
  { 57542, UINT64_C(1891698992) },
  { 122148, UINT64_C(4004515805) },
  { 46254, UINT64_C(1521227949) },
  { 42850, UINT64_C(1408996941) },
  { 92661, UINT64_C(3037901517) },
  { 57720, UINT64_C(1897369989) },
};

static void
test_crypto_ope_vectors(void *arg)
{
  (void)arg;
  uint8_t key[32];
  crypto_ope_t *ope = NULL, *ope2 = NULL;

  base16_decode((char*)key, 32, OPE_TEST_KEY, strlen(OPE_TEST_KEY));

  ope = crypto_ope_new(key);
  key[8] += 1;
  ope2 = crypto_ope_new(key);
  unsigned i;
  for (i = 0; i < ARRAY_LENGTH(OPE_TEST_VECTORS); ++i) {
    int val = OPE_TEST_VECTORS[i].v;
    uint64_t res = OPE_TEST_VECTORS[i].r;

    tt_u64_op(crypto_ope_encrypt(ope, val), OP_EQ, res);
    tt_u64_op(crypto_ope_encrypt(ope2, val), OP_NE, res);
  }
 done:
  crypto_ope_free(ope);
  crypto_ope_free(ope2);
}

struct testcase_t crypto_ope_tests[] = {
  { "consistency", test_crypto_ope_consistency, 0, NULL, NULL },
  { "oob", test_crypto_ope_oob, 0, NULL, NULL },
  { "vectors", test_crypto_ope_vectors, 0, NULL, NULL },
  END_OF_TESTCASES
};
