/*
 * Perform both rotation and reframing of all the image of the stream
 * This file is part of ufo-serge filter set.
 * Copyright (C) 2021 Serge Cohen
 *
 * This file is part of Ufo.
 *
 * This library is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Serge Cohen <serge.cohen@ipanema-remote.fr>
 */
#include "config.h"

#include <glib.h>
#include <math.h>

#include <ufo/ufo-cl.h>

// From UFO, to manage addressing and interpolation (OpenCL sampler)
#include "../src/common/ufo-addressing.h"
#include "../src/common/ufo-interpolation.h"

#include "ufo-rotate-reframe-task.h"


struct _UfoRotateReframeTaskPrivate {
  gfloat angle;
  gfloat rot_center[2];
  gfloat new_center[2];
  gint new_frame[2];
  AddressingMode addressing_mode;
  Interpolation interpolation;

  cl_context context;
  cl_kernel kernel;
  cl_sampler sampler;
  gfloat sincos[2];
};

static void ufo_task_interface_init (UfoTaskIface *iface);

G_DEFINE_TYPE_WITH_CODE (UfoRotateReframeTask, ufo_rotate_reframe_task, UFO_TYPE_TASK_NODE,
                         G_ADD_PRIVATE (UfoRotateReframeTask);
			  G_IMPLEMENT_INTERFACE (UFO_TYPE_TASK,
						 ufo_task_interface_init))

#define UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(obj) ufo_rotate_reframe_task_get_instance_private ((UfoRotateReframeTask *)obj)

/*
G_DEFINE_TYPE_WITH_CODE (UfoRotateReframeTask, ufo_rotate_reframe_task, UFO_TYPE_TASK_NODE,
                         G_IMPLEMENT_INTERFACE (UFO_TYPE_TASK,
                                                ufo_task_interface_init))

#define UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), UFO_TYPE_ROTATE_REFRAME_TASK, UfoRotateReframeTaskPrivate))
*/


  enum {
  PROP_0,
  PROP_ANGLE,
  PROP_ROT_CENTER,
  PROP_NEW_CENTER,
  PROP_NEW_FRAME,
  PROP_INTERPOLATION,
  PROP_ADDRESSING_MODE,
  N_PROPERTIES
};

static GParamSpec *properties[N_PROPERTIES] = { NULL, };

UfoNode *
ufo_rotate_reframe_task_new (void)
{
  return UFO_NODE (g_object_new (UFO_TYPE_ROTATE_REFRAME_TASK, NULL));
}

static void
ufo_rotate_reframe_task_setup (UfoTask *task,
			       UfoResources *resources,
			       GError **error)
{
  UfoRotateReframeTaskPrivate *priv;
  cl_int cl_error;

  priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(task);

  priv->context = ufo_resources_get_context (resources);
  priv->kernel = ufo_resources_get_kernel (resources, "rotate-reframe.cl", "rotate_reframe", NULL, error);

  if (NULL != priv->kernel)
    UFO_RESOURCES_CHECK_SET_AND_RETURN (clRetainKernel (priv->kernel), error);

  // Getting the sampler object, non normalised coordinates :
  priv->sampler = clCreateSampler (priv->context, (cl_bool)FALSE, priv->addressing_mode, priv->interpolation, &cl_error);

  // Compute the sinus and cosinus of the angle
  priv->sincos[0] = sin(priv->angle);
  priv->sincos[1] = cos(priv->angle);

}

static void
ufo_rotate_reframe_task_get_requisition (UfoTask *task,
					 UfoBuffer **inputs,
					 UfoRequisition *requisition,
					 GError **error)
{
  UfoRotateReframeTaskPrivate *priv;
  UfoRequisition in_req;

  priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(task);
  ufo_buffer_get_requisition (inputs[0], &in_req);

  // If rot_center is not set : put it in the center of the input frame:
  if ( G_MAXFLOAT == priv->rot_center[0] ) {
    priv->rot_center[0] = (float)(in_req.dims[0]) / 2.0f;
  }
  if ( G_MAXFLOAT == priv->rot_center[1] ) {
    priv->rot_center[1] = (float)(in_req.dims[1]) / 2.0f;
  }

  // If any of the new frame is <= 0, then use the input's value :
  if ( 0 >=  priv->new_frame[0] ) {
    priv->new_frame[0] = in_req.dims[0];
  }
  if ( 0 >=  priv->new_frame[1] ) {
    priv->new_frame[1] = in_req.dims[1];
  }

  // If rot_center is not set : put it in the center of the input frame:
  if ( -G_MAXFLOAT == priv->new_center[0] ) {
    priv->new_center[0] = (float)(priv->new_frame[0]) / 2.0f;
  }
  if ( -G_MAXFLOAT == priv->new_center[1] ) {
    priv->new_center[1] = (float)(priv->new_frame[1]) / 2.0f;
  }

  // The output is precisely the new_frame :
  requisition->n_dims = 2;
  requisition->dims[0] = priv->new_frame[0];
  requisition->dims[1] = priv->new_frame[1];
}

static guint
ufo_rotate_reframe_task_get_num_inputs (UfoTask *task)
{
  return 1;
}

static guint
ufo_rotate_reframe_task_get_num_dimensions (UfoTask *task,
					    guint input)
{
  return 2;
}

static UfoTaskMode
ufo_rotate_reframe_task_get_mode (UfoTask *task)
{
  return UFO_TASK_MODE_PROCESSOR | UFO_TASK_MODE_GPU;
}

static gboolean
ufo_rotate_reframe_task_process (UfoTask *task,
				 UfoBuffer **inputs,
				 UfoBuffer *output,
				 UfoRequisition *requisition)
{
  UfoRotateReframeTaskPrivate *priv;
  UfoProfiler *profiler;
  UfoGpuNode *node;
  cl_command_queue cmd_queue;
  cl_mem in_mem;
  cl_mem out_mem;

  priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(task);
  node = UFO_GPU_NODE (ufo_task_node_get_proc_node (UFO_TASK_NODE (task)));
  cmd_queue = ufo_gpu_node_get_cmd_queue (node);
  in_mem = ufo_buffer_get_device_image (inputs[0], cmd_queue);
  out_mem = ufo_buffer_get_device_image (output, cmd_queue);

  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 0, sizeof (cl_mem), &out_mem));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 1, sizeof (cl_mem), &in_mem));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 2, sizeof (cl_sampler), &priv->sampler));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 3, sizeof (cl_float2), priv->sincos));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 4, sizeof (cl_float2), priv->rot_center));
  UFO_RESOURCES_CHECK_CLERR (clSetKernelArg (priv->kernel, 5, sizeof (cl_float2), priv->new_center));

  profiler = ufo_task_node_get_profiler (UFO_TASK_NODE (task));
  ufo_profiler_call (profiler, cmd_queue, priv->kernel, 2, requisition->dims, NULL);

  return TRUE;
}


static void
ufo_rotate_reframe_task_set_property (GObject *object,
				      guint property_id,
				      const GValue *value,
				      GParamSpec *pspec)
{
  UfoRotateReframeTaskPrivate *priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE (object);
  GValueArray *array;

  switch (property_id) {
  case PROP_ANGLE:
    priv->angle = g_value_get_float(value);
    priv->sincos[0] = sin(priv->angle);
    priv->sincos[1] = cos(priv->angle);
    break;
  case PROP_ROT_CENTER:
    array = (GValueArray *) g_value_get_boxed(value);
    priv->rot_center[0] = g_value_get_float(g_value_array_get_nth(array, 0));
    priv->rot_center[1] = g_value_get_float(g_value_array_get_nth(array, 1));
    break;
  case PROP_NEW_CENTER:
    array = (GValueArray *) g_value_get_boxed(value);
    priv->new_center[0] = g_value_get_float(g_value_array_get_nth(array, 0));
    priv->new_center[1] = g_value_get_float(g_value_array_get_nth(array, 1));
    break;
  case PROP_NEW_FRAME :
    array = (GValueArray *) g_value_get_boxed(value);
    priv->new_frame[0] = g_value_get_int(g_value_array_get_nth(array, 0));
    priv->new_frame[1] = g_value_get_int(g_value_array_get_nth(array, 1));
    break;
  case PROP_INTERPOLATION:
    priv->interpolation = g_value_get_enum(value);
    break;
  case PROP_ADDRESSING_MODE:
    priv->addressing_mode = g_value_get_enum(value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}

static void
ufo_rotate_reframe_task_get_property (GObject *object,
				      guint property_id,
				      GValue *value,
				      GParamSpec *pspec)
{
  UfoRotateReframeTaskPrivate *priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE (object);
  GValueArray *array;
  GValue x = G_VALUE_INIT;

  switch (property_id) {
  case PROP_ANGLE:
    g_value_set_float (value, priv->angle);
    break;
  case PROP_ROT_CENTER:
    array = g_value_array_new (2);
    g_value_init(&x, G_TYPE_FLOAT);
    g_value_set_float(&x, priv->rot_center[0]);
    g_value_array_append(array, &x);
    g_value_set_float(&x, priv->rot_center[1]);
    g_value_array_append(array, &x);
    g_value_take_boxed(value, array);
    break;
  case PROP_NEW_CENTER:
    array = g_value_array_new (2);
    g_value_init(&x, G_TYPE_FLOAT);
    g_value_set_float(&x, priv->new_center[0]);
    g_value_array_append(array, &x);
    g_value_set_float(&x, priv->new_center[1]);
    g_value_array_append(array, &x);
    g_value_take_boxed(value, array);
    break;
  case PROP_NEW_FRAME :
    array = g_value_array_new (2);
    g_value_init(&x, G_TYPE_FLOAT);
    g_value_set_float(&x, priv->new_frame[0]);
    g_value_array_append(array, &x);
    g_value_set_float(&x, priv->new_frame[1]);
    g_value_array_append(array, &x);
    g_value_take_boxed(value, array);
    break;
  case PROP_INTERPOLATION:
    g_value_set_enum(value, priv->interpolation);
    break;
  case PROP_ADDRESSING_MODE:
    g_value_set_enum(value, priv->addressing_mode);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}

static void
ufo_rotate_reframe_task_finalize (GObject *object)
{
  UfoRotateReframeTaskPrivate *priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE (object);

  if (priv->kernel) {
    UFO_RESOURCES_CHECK_CLERR (clReleaseKernel (priv->kernel));
    priv->kernel = NULL;
  }

  if (priv->sampler) {
    UFO_RESOURCES_CHECK_CLERR (clReleaseSampler (priv->sampler));
    priv->sampler = NULL;
  }

  if (priv->context) {
    UFO_RESOURCES_CHECK_CLERR (clReleaseContext (priv->context));
    priv->context = NULL;
  }

  G_OBJECT_CLASS (ufo_rotate_reframe_task_parent_class)->finalize (object);
}

static void
ufo_task_interface_init (UfoTaskIface *iface)
{
  iface->setup = ufo_rotate_reframe_task_setup;
  iface->get_num_inputs = ufo_rotate_reframe_task_get_num_inputs;
  iface->get_num_dimensions = ufo_rotate_reframe_task_get_num_dimensions;
  iface->get_mode = ufo_rotate_reframe_task_get_mode;
  iface->get_requisition = ufo_rotate_reframe_task_get_requisition;
  iface->process = ufo_rotate_reframe_task_process;
}

static void
ufo_rotate_reframe_task_class_init (UfoRotateReframeTaskClass *klass)
{
  GObjectClass *oclass = G_OBJECT_CLASS (klass);

  oclass->set_property = ufo_rotate_reframe_task_set_property;
  oclass->get_property = ufo_rotate_reframe_task_get_property;
  oclass->finalize = ufo_rotate_reframe_task_finalize;

  GParamSpec *xy_vals = g_param_spec_float ("float-coord-values",
					    "Float coordinates values",
					    "",
					    -G_MAXFLOAT,
					    G_MAXFLOAT,
					    -G_MAXFLOAT,
					    G_PARAM_READWRITE);


  GParamSpec *frame_vals = g_param_spec_int ("int-frame-values",
					    "Dimension of a frame, int values",
					    "",
					    -1,
					    G_MAXINT,
					    -1,
					    G_PARAM_READWRITE);

  properties[PROP_ANGLE] =
    g_param_spec_float ("angle",
			"Rotation angle in radians",
			"Rotation angle in radians",
			-G_MAXFLOAT, G_MAXFLOAT, 0.0f,
			G_PARAM_READWRITE);

  properties[PROP_ROT_CENTER] =
    g_param_spec_value_array ("rot-center",
			      "Center of rotation (x, y)",
			      "Center of rotation (x, y) in the input frame",
			      xy_vals,
			      G_PARAM_READWRITE);

  properties[PROP_NEW_CENTER] =
    g_param_spec_value_array ("new-center",
			      "Center of rotation (x, y)",
			      "Center of rotation (x, y) in the new frame",
			      xy_vals,
			      G_PARAM_READWRITE);

  properties[PROP_NEW_FRAME] =
    g_param_spec_value_array ("new-frame",
			      "Dimensions of the new frame (x, y)",
			      "Dimensions of the new frame as (x, y) pixels",
			      frame_vals,
			      G_PARAM_READWRITE);


  properties[PROP_ADDRESSING_MODE] =
    g_param_spec_enum ("addressing-mode",
		       "Outlier treatment (\"none\", \"clamp\", \"clamp_to_edge\", \"repeat\", \"mirrored_repeat\"). Defaulting to clamp_to_edge",
		       "Outlier treatment (\"none\", \"clamp\", \"clamp_to_edge\", \"repeat\", \"mirrored_repeat\"). Defaulting to clamp_to_edge",
		       g_enum_register_static ("ufo_rot_ref_addressing_mode", addressing_values),
		       CL_ADDRESS_CLAMP,
		       G_PARAM_READWRITE);

  properties[PROP_INTERPOLATION] =
    g_param_spec_enum ("interpolation",
		       "Interpolation (\"nearest\" or \"linear\"). Defaulting to linear",
		       "Interpolation (\"nearest\" or \"linear\"). Defaulting to linear",
		       g_enum_register_static ("ufo_rot_ref_interpolation", interpolation_values),
		       CL_FILTER_LINEAR,
		       G_PARAM_READWRITE);



  for (guint i = PROP_0 + 1; i < N_PROPERTIES; i++)
    g_object_class_install_property (oclass, i, properties[i]);

  // No more needed : using the new G_ADD_PRIVATE macro
  //  g_type_class_add_private (oclass, sizeof(UfoRotateReframeTaskPrivate));
}

static void
ufo_rotate_reframe_task_init(UfoRotateReframeTask *self)
{
  self->priv = UFO_ROTATE_REFRAME_TASK_GET_PRIVATE(self);
  self->priv->angle = 0;
  self->priv->sincos[0] = sin(0.0f);
  self->priv->sincos[1] = cos(0.0f);
  self->priv->rot_center[0] = G_MAXFLOAT;
  self->priv->rot_center[1] = G_MAXFLOAT;
  self->priv->new_center[0] = -G_MAXFLOAT;
  self->priv->new_center[1] = -G_MAXFLOAT;
  self->priv->new_frame[0] = -1;
  self->priv->new_frame[1] = -1;
  self->priv->addressing_mode = CL_ADDRESS_CLAMP;
  self->priv->interpolation = CL_FILTER_LINEAR;
}
