/*******************************************************************************
 * Part of "Intel(R) Active Management Technology (Intel(R) AMT)
 *                   User Notification Service (UNS)"
 *
 * Copyright (c) 2007 Intel Corp.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions, and the following disclaimer,
 *    without modification.
 * 2. Redistributions in binary form must reproduce at minimum a disclaimer
 *    substantially similar to the "NO WARRANTY" disclaimer below
 *    ("Disclaimer") and any redistribution must be conditioned upon
 *    including a substantially similar Disclaimer requirement for further
 *    binary redistribution.
 * 3. Neither the names of the above-listed copyright holders nor the names
 *    of any contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * NO WARRANTY
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       UNSMainLinux.cpp
//
//  Contents:   The Linux main UNS application.
//
//----------------------------------------------------------------------------

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <signal.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <errno.h>
#include "UserNotificationServiceLinux.h"
#include "UNSCommonDefs.h"
#include "Options.h"
#include "Thread.h"

#ifdef DAEMON
#include "daemonize.h"
#endif //DAEMON

#define LOCK_PATH "/var/run/"
/* Change this to whatever your daemon is called */
#define DAEMON_PID_FILE "uns.pid"

// Global variable, the main service.
UserNotificationService service;
int lock_pid_file_fd = -1;

// Handler for termination signal.
void terminationHandler (int signum, siginfo_t* sinfo, void* dummy)
{
	service.Stop();
}
//Signals action register
void setSignalsHandler()
{
        int sigSet = 0;
        // Termination signal handler.
        struct sigaction terminateAction;
        // Set up the structure to specify the termination action.
        terminateAction.sa_sigaction = terminationHandler;
        sigemptyset(&terminateAction.sa_mask);
        terminateAction.sa_flags = SA_SIGINFO;
        sigSet &= sigaction (SIGTERM, &terminateAction, NULL);
        sigSet &= sigaction (SIGQUIT, &terminateAction, NULL);
        sigSet &= sigaction (SIGHUP, &terminateAction, NULL);
        sigSet &= sigaction (SIGINT,  &terminateAction, NULL);
        sigSet &= sigaction (SIGPIPE, &terminateAction, NULL);
        sigSet &= sigaction (SIGALRM, &terminateAction, NULL);
        sigSet &= sigaction (SIGUSR1, &terminateAction, NULL);
        sigSet &= sigaction (SIGUSR2, &terminateAction, NULL);

  	if (sigSet != 0) {
		printf ("Warning: failed to register terminate signal handler\n");
	}
}

/**
 *  lock_pid_file - creates a pid file and writes current process pid into it
 * 
 *  lockfile - name of a file to be created
 * 
 *  return: 0 on success, -1 on fatal error, -2 on error
 **/
int lock_pid_file(const char* lockfile)
{
	int lfp = -1;
	size_t towrite = 0;
	ssize_t written = 0;
	int error = 0;
	int haserror = 0;
	char pid_buf[32];

	/* Create the lock file as the current user */
	if (lockfile && lockfile[0]) {
		lfp = open(lockfile, O_RDWR | O_CREAT, 0644);
		if (lfp < 0) {
			syslog(LOG_ERR,
			       "unable to create lock file %s, code=%d (%s)",
			       lockfile, errno, strerror(errno));
			return -1;
		}
		if (-1 == flock(lfp, LOCK_EX | LOCK_NB)) {
			error = errno;
			if (EWOULDBLOCK == errno) {
				syslog(LOG_ERR, "The UNS service is already running!");
				close(lfp);
			} else {
				syslog(LOG_ERR,
				       "unable to lock file %s, code=%d (%s)",
				       lockfile, error, strerror(error));
				close(lfp);
				unlink(lockfile);
				return -2;
			}
			return -1;
		}
		if (-1 == ftruncate(lfp, 0)) {
			syslog(LOG_ERR,
			       "unable to fruncate lock file %s, code=%d (%s)",
			       lockfile, errno, strerror(errno));
			close(lfp);
			unlink(lockfile);
			return -2;
		}
		snprintf(pid_buf, sizeof(pid_buf), "%u", getpid());
		towrite = strnlen(pid_buf, 31);
		written = write(lfp, pid_buf, towrite);
		if (-1 == written) {
			error = errno;
			haserror = 1;
		} else if (towrite != (size_t)written) {
			haserror = 1;
		} else if (-1 == fsync(lfp)) {
			error = errno;
			haserror = 1;
		}
		if (1 == haserror) {
			syslog(LOG_ERR,
			       "unable to write pid into lock file %s, code=%d (%s)",
			       lockfile, error, strerror(error));
			close(lfp);
			unlink(lockfile);
			return -2;
		}
		lock_pid_file_fd = lfp;
	}
	return 0;
}

/**
 *  unlock_pid_file - removes a pid file
 * 
 *  lockfile - name of a file to be removed
 **/
void unlock_pid_file(const char* lockfile)
{
	if (-1 != lock_pid_file_fd) {
		close(lock_pid_file_fd);
		lock_pid_file_fd = -1;
		unlink(lockfile);
	}
}


// Main service entry.
int main(int argc, const char *argv[])
{
	int lockresult = -1;
	// Check for client/server remote debug IP.
	unsigned long error;

	if (argc > 1) {
		lockresult = lock_pid_file(LOCK_PATH DAEMON_PID_FILE);
		if (-2 == lockresult) {
			lockresult = lock_pid_file(LOCK_PATH DAEMON_PID_FILE);
		}
		if (0 != lockresult) {
			printf("Intel(R) AMT User Notification unable to lock pidfile. Is UNS already running?\n");
			return EEXIST;
		}

		service.SetCredentialsMode(argc, argv, &error);

		unlock_pid_file(LOCK_PATH DAEMON_PID_FILE);

		return error;
	}

#ifdef DAEMON
	daemonize();
#endif //DAEMON
    
	if (service.ServiceInitialization(argc, argv, &error) != 0) {
		return error;
	}

	lockresult = lock_pid_file(LOCK_PATH DAEMON_PID_FILE);
	if (-2 == lockresult) {
		lockresult = lock_pid_file(LOCK_PATH DAEMON_PID_FILE);
	}
	if (0 != lockresult) {
		return EEXIST;
	}

	setSignalsHandler();

	// Run the service in a new thread.
	service.Start();

	unlock_pid_file(LOCK_PATH DAEMON_PID_FILE);

	return 0;
}
