#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdbool.h>
#include <ctype.h>
#include <assert.h>

#include <rtosc/rtosc.h>
#include <rtosc/arg-val-math.h>

const char *rtosc_argument_string(const char *msg)
{
    assert(msg && *msg);
    while(*++msg); //skip pattern
    while(!*++msg);//skip null
    return msg+1;  //skip comma
}

unsigned rtosc_narguments(const char *msg)
{
    const char *args = rtosc_argument_string(msg);
    int nargs = 0;
    while(*args++)
        nargs += (*args == ']' || *args == '[') ? 0 : 1;
    return nargs;
}

static int has_reserved(char type)
{
    switch(type)
    {
        case 'i'://official types
        case 's':
        case 'b':
        case 'f':

        case 'h'://unofficial
        case 't':
        case 'd':
        case 'S':
        case 'r':
        case 'm':
        case 'c':
            return 1;
        case 'T':
        case 'F':
        case 'N':
        case 'I':
        case '[':
        case ']':
            return 0;
    }

    //Should not happen
    return 0;
}

static unsigned nreserved(const char *args)
{
    unsigned res = 0;
    for(;*args;++args)
        res += has_reserved(*args);

    return res;
}

char rtosc_type(const char *msg, unsigned nargument)
{
    assert(nargument < rtosc_narguments(msg));
    const char *arg = rtosc_argument_string(msg);
    while(1) {
        if(*arg == '[' || *arg == ']')
            ++arg;
        else if(!nargument || !*arg)
            return *arg;
        else
            ++arg, --nargument;
    }
}

static unsigned arg_start(const char *msg_)
{
    const uint8_t *msg = (const uint8_t*)msg_;
    //Iterate to the right position
    const uint8_t *args = (const uint8_t*) rtosc_argument_string(msg_);
    const uint8_t *aligned_ptr = args-1;
    const uint8_t *arg_pos = args;

    while(*++arg_pos);
    //Alignment
    arg_pos += 4-(arg_pos-aligned_ptr)%4;
    return arg_pos-msg;
}

static unsigned arg_size(const uint8_t *arg_mem, char type)
{
    if(!has_reserved(type))
        return 0;
    const uint8_t  *arg_pos=arg_mem;
    uint32_t blob_length = 0;
    switch(type)
    {
        case 'h':
        case 't':
        case 'd':
            return 8;
        case 'm':
        case 'r':
        case 'f':
        case 'c':
        case 'i':
            return 4;
        case 'S':
        case 's':
            while(*++arg_pos);
            arg_pos += 4-(arg_pos-arg_mem)%4;
            return arg_pos-arg_mem;
        case 'b':
            blob_length |= (*arg_pos++ << 24);
            blob_length |= (*arg_pos++ << 16);
            blob_length |= (*arg_pos++ << 8);
            blob_length |= (*arg_pos++);
            if(blob_length%4)
                blob_length += 4-blob_length%4;
            arg_pos += blob_length;
            return arg_pos-arg_mem;
        default:
            assert("Invalid Type");
    }
    return -1;
}

static unsigned arg_off(const char *msg, unsigned idx)
{
    if(!has_reserved(rtosc_type(msg,idx)))
        return 0;

    //Iterate to the right position
    const uint8_t *args = (const uint8_t*) rtosc_argument_string(msg);
    const uint8_t *aligned_ptr = args-1;
    const uint8_t *arg_pos = args;

    while(*++arg_pos);
    //Alignment
    arg_pos += 4-(arg_pos-((uint8_t*)aligned_ptr))%4;

    //ignore any leading '[' or ']'
    while(*args == '[' || *args == ']')
        ++args;

    while(idx--) {
        char type = *args++;
        if(type == '[' || type == ']')
            idx++;//not a valid arg idx
        else
            arg_pos += arg_size(arg_pos, type);
    }
    return arg_pos-(uint8_t*)msg;
}

size_t rtosc_message(char   *buffer,
                     size_t      len,
                     const char *address,
                     const char *arguments,
                     ...)
{
    va_list va;
    va_start(va, arguments);
    size_t result = rtosc_vmessage(buffer, len, address, arguments, va);
    va_end(va);
    return result;
}

//Calculate the size of the message without writing to a buffer
static size_t vsosc_null(const char        *address,
                         const char        *arguments,
                         const rtosc_arg_t *args)
{
    unsigned pos = 0;
    pos += strlen(address);
    pos += 4-pos%4;//get 32 bit alignment
    pos += 1+strlen(arguments);
    pos += 4-pos%4;

    unsigned toparse = nreserved(arguments);
    unsigned arg_pos = 0;

    //Take care of varargs
    while(toparse)
    {
        char arg = *arguments++;
        assert(arg);
        int i;
        const char *s;
        switch(arg) {
            case 'h':
            case 't':
            case 'd':
                ++arg_pos;
                pos += 8;
                --toparse;
                break;
            case 'm':
            case 'r':
            case 'c':
            case 'f':
            case 'i':
                ++arg_pos;
                pos += 4;
                --toparse;
                break;
            case 's':
            case 'S':
                s = args[arg_pos++].s;
                assert(s && "Input strings CANNOT be NULL");
                pos += strlen(s);
                pos += 4-pos%4;
                --toparse;
                break;
            case 'b':
                i = args[arg_pos++].b.len;
                pos += 4 + i;
                if(pos%4)
                    pos += 4-pos%4;
                --toparse;
                break;
            default:
                ;
        }
    }

    return pos;
}

static const rtosc_cmp_options default_cmp_options = { 0.0 };

void rtosc_arg_val_itr_init(rtosc_arg_val_t_const_itr* itr,
                            const rtosc_arg_val_t* av)
{
    itr->av = av;
    itr->i = itr->range_i = 0;
}

//! helper function for arg val comparing
//! this usually just returns the value from operand, except for range operands,
//! where the value is being interpolated
const rtosc_arg_val_t* rtosc_arg_val_itr_get(
    const rtosc_arg_val_t_const_itr* itr,
    rtosc_arg_val_t* buffer)
{
    const rtosc_arg_val_t* result;
    if(itr->av->type == '-')
    {
        if(itr->av->val.r.has_delta)
            rtosc_arg_val_range_arg(itr->av, itr->range_i, buffer);
        else
            *buffer = itr->av[1];
        result = buffer;
    }
    else result = itr->av;
    return result;
}

void rtosc_arg_val_itr_next(rtosc_arg_val_t_const_itr* itr)
{
    // increase the range index
    if(itr->av->type == '-')
    {
        // increase if the limit was reached, and the limit was not infinity (=0)
        if(++itr->range_i >= itr->av->val.r.num && itr->av->val.r.num)
        {
            if(itr->av->val.r.has_delta)
            {
                ++itr->av;
                ++itr->i;
            }
            ++itr->av;
            ++itr->i;
            itr->range_i = 0;
        }
    }

    // if not inside a range (or at its beginning), increase the index
    if(!itr->range_i)
    {
        if(itr->av->type == 'a')
        {
            itr->i += itr->av->val.a.len;
            itr->av += itr->av->val.a.len;
        }
        ++itr->i;
        ++itr->av;
    }
}

// abort only if one side was finished, or if both are infinite ranges
int rtosc_arg_vals_cmp_has_next(const rtosc_arg_val_t_const_itr* litr,
                                const rtosc_arg_val_t_const_itr* ritr,
                                size_t lsize, size_t rsize)
{
    return     (litr->i < lsize) && (ritr->i < rsize)
            && (litr->av->type != '-' || ritr->av->type != '-' ||
                litr->av->val.r.num || ritr->av->val.r.num);
}

// arrays are equal by now, but is one longer?
// each array must have either been completely passed, or it must
// have reached an infinite range
int rtosc_arg_vals_eq_after_abort(const rtosc_arg_val_t_const_itr* litr,
                                  const rtosc_arg_val_t_const_itr* ritr,
                                  size_t lsize, size_t rsize)
{
    return    (litr->i == lsize ||
                  (litr->av->type == '-' && !litr->av->val.r.num))
           && (ritr->i == rsize ||
                  (ritr->av->type == '-' && !ritr->av->val.r.num));
}

// compare single elements, ranges excluded
int rtosc_arg_vals_eq_single(const rtosc_arg_val_t* _lhs,
                             const rtosc_arg_val_t* _rhs,
                             const rtosc_cmp_options* opt)
{
#define mfabs(val) (((val) >= 0) ? (val) : -(val))
    int rval;

    if(!opt)
        opt = &default_cmp_options;

    if(_lhs->type == _rhs->type)
    switch(_lhs->type)
    {
        case 'i':
        case 'c':
        case 'r':
            rval = _lhs->val.i == _rhs->val.i;
            break;
        case 'I':
        case 'T':
        case 'F':
        case 'N':
            rval = 1;
            break;
        case 'f':
            rval = (opt->float_tolerance == 0.0)
                   ? _lhs->val.f == _rhs->val.f
                   : mfabs(_lhs->val.f - _rhs->val.f) <=
                   (float)opt->float_tolerance;
            break;
        case 'd':
            rval = (opt->float_tolerance == 0.0)
                   ? _lhs->val.d == _rhs->val.d
                   : mfabs(_lhs->val.d - _rhs->val.d) <=
                   opt->float_tolerance;
            break;
        case 'h':
            rval = _lhs->val.h == _rhs->val.h;
            break;
        case 't':
            rval = _lhs->val.t == _rhs->val.t;
            break;
        case 'm':
            rval = 0 == memcmp(_lhs->val.m, _rhs->val.m, 4);
            break;
        case 's':
        case 'S':
            rval = (_lhs->val.s == NULL || _rhs->val.s == NULL)
                 ? _lhs->val.s == _rhs->val.s
                 : (0 == strcmp(_lhs->val.s, _rhs->val.s));
            break;
        case 'b':
        {
            int32_t lbs = _lhs->val.b.len,
                    rbs = _rhs->val.b.len;
            rval = lbs == rbs;
            if(rval)
                rval = 0 == memcmp(_lhs->val.b.data, _rhs->val.b.data, lbs);
            break;
        }
        case 'a':
        {
            if(     _lhs->val.a.type != _rhs->val.a.type
               && !(_lhs->val.a.type == 'T' && _rhs->val.a.type == 'F')
               && !(_lhs->val.a.type == 'F' && _rhs->val.a.type == 'T'))
                rval = 0;
            else
                rval = rtosc_arg_vals_eq(_lhs+1, _rhs+1,
                                         _lhs->val.a.len, _rhs->val.a.len,
                                         opt);
            break;
        }
        case '-':
            assert(false);
            break;
    }
    else
    {
        rval = 0;
    }
    return rval;
#undef mfabs
}

int rtosc_arg_vals_eq(const rtosc_arg_val_t* lhs, const rtosc_arg_val_t* rhs,
                      size_t lsize, size_t rsize,
                      const rtosc_cmp_options* opt)
{
    // used if the value of lhs or rhs is range-computed:
    rtosc_arg_val_t rlhs, rrhs;

    rtosc_arg_val_t_const_itr litr, ritr;
    rtosc_arg_val_itr_init(&litr, lhs);
    rtosc_arg_val_itr_init(&ritr, rhs);

    int rval = 1;

    if(!opt)
        opt = &default_cmp_options;

    for( ; rtosc_arg_vals_cmp_has_next(&litr, &ritr, lsize, rsize) && rval;
        rtosc_arg_val_itr_next(&litr),
        rtosc_arg_val_itr_next(&ritr))
    {
        rval = rtosc_arg_vals_eq_single(rtosc_arg_val_itr_get(&litr, &rlhs),
                                        rtosc_arg_val_itr_get(&ritr, &rrhs),
                                        opt);
    }

    return rval
        ? rtosc_arg_vals_eq_after_abort(&litr, &ritr, lsize, rsize)
        : rval;
}

// compare single elements, ranges excluded
int rtosc_arg_vals_cmp_single(const rtosc_arg_val_t* _lhs,
                              const rtosc_arg_val_t* _rhs,
                              const rtosc_cmp_options* opt)
{
#define cmp_3way(val1,val2) (((val1) == (val2)) \
                            ? 0 \
                            : (((val1) > (val2)) ? 1 : -1))
#define mfabs(val) (((val) >= 0) ? (val) : -(val))

    int rval;

    if(!opt)
        opt = &default_cmp_options;

    if(_lhs->type == _rhs->type)
    switch(_lhs->type)
    {
        case 'i':
        case 'c':
        case 'r':
            rval = cmp_3way(_lhs->val.i, _rhs->val.i);
            break;
        case 'I':
        case 'T':
        case 'F':
        case 'N':
            rval = 0;
            break;
        case 'f':
            rval = (opt->float_tolerance == 0.0)
                   ? cmp_3way(_lhs->val.f, _rhs->val.f)
                   : (mfabs(_lhs->val.f - _rhs->val.f)
                      <= (float)opt->float_tolerance)
                      ? 0
                      : ((_lhs->val.f > _rhs->val.f) ? 1 : -1);
            break;
        case 'd':
            rval = (opt->float_tolerance == 0.0)
                   ? cmp_3way(_lhs->val.d, _rhs->val.d)
                   : (mfabs(_lhs->val.d - _rhs->val.d)
                      <= opt->float_tolerance)
                      ? 0
                      : ((_lhs->val.d > _rhs->val.d) ? 1 : -1);
            break;
        case 'h':
            rval = cmp_3way(_lhs->val.h, _rhs->val.h);
            break;
        case 't':
            // immediately is considered lower than everything else
            // this means if you send two events to a client,
            // one being "immediately" and one being different, the
            // immediately-event has the higher priority, event if the
            // other one is in the past
            rval = (_lhs->val.t == 1)
                   ? (_rhs->val.t == 1)
                     ? 0
                     : -1 // _lhs has higher priority => _lhs < _rhs
                   : (_rhs->val.t == 1)
                     ? 1
                     : cmp_3way(_lhs->val.t, _rhs->val.t);
            break;
        case 'm':
            rval = memcmp(_lhs->val.m, _rhs->val.m, 4);
            break;
        case 's':
        case 'S':
            rval = (_lhs->val.s == NULL || _rhs->val.s == NULL)
                 ? cmp_3way(_lhs->val.s, _rhs->val.s)
                 : strcmp(_lhs->val.s, _rhs->val.s);
            break;
        case 'b':
        {
            int32_t lbs = _lhs->val.b.len,
                    rbs = _rhs->val.b.len;
            int32_t minlen = (lbs < rbs) ? lbs : rbs;
            rval = memcmp(_lhs->val.b.data, _rhs->val.b.data, minlen);
            if(lbs != rbs && !rval)
            {
                // both equal until here
                // the string that ends here is lexicographically smaller
                rval = (lbs > rbs)
                       ? _lhs->val.b.data[minlen]
                       : -_rhs->val.b.data[minlen];
            }

            break;
        }
        case 'a':
        {
            int32_t llen = _lhs->val.a.len, rlen = _rhs->val.a.len;
            if(     _lhs->val.a.type != _rhs->val.a.type
               && !(_lhs->val.a.type == 'T' && _rhs->val.a.type == 'F')
               && !(_lhs->val.a.type == 'F' && _rhs->val.a.type == 'T'))
                rval = (_lhs->val.a.type > _rhs->val.a.type) ? 1 : -1;
            else
            {
                // the arg vals differ in this array => compare and return
                rval = rtosc_arg_vals_cmp(_lhs+1, _rhs+1, llen, rlen, opt);
            }
            break;
        }
        case '-':
            assert(false);
            break;
    }
    else
    {
        rval = (_lhs->type > _rhs->type) ? 1 : -1;
    }

    return rval;

#undef mfabs
#undef cmp_3way
}

int rtosc_arg_vals_cmp(const rtosc_arg_val_t* lhs, const rtosc_arg_val_t* rhs,
                       size_t lsize, size_t rsize,
                       const rtosc_cmp_options* opt)
{
    // used if the value of lhs or rhs is range-computed:
    rtosc_arg_val_t rlhs, rrhs;

    rtosc_arg_val_t_const_itr litr, ritr;
    rtosc_arg_val_itr_init(&litr, lhs);
    rtosc_arg_val_itr_init(&ritr, rhs);

    int rval = 0;

    if(!opt)
        opt = &default_cmp_options;

    for( ; rtosc_arg_vals_cmp_has_next(&litr, &ritr, lsize, rsize) && !rval;
        rtosc_arg_val_itr_next(&litr),
        rtosc_arg_val_itr_next(&ritr))
    {
        rval = rtosc_arg_vals_cmp_single(rtosc_arg_val_itr_get(&litr, &rlhs),
                                         rtosc_arg_val_itr_get(&ritr, &rrhs),
                                         opt);
    }

    return rval ? rval
                : (rtosc_arg_vals_eq_after_abort(&litr, &ritr, lsize, rsize))
                    ? 0
                    // they're equal until here, so one array must have
                    // elements left:
                    : (lsize-(litr.i) > rsize-(ritr.i))
                        ? 1
                        : -1;
}

void rtosc_v2args(rtosc_arg_t* args, size_t nargs, const char* arg_str,
                  rtosc_va_list_t* ap)
{
    unsigned arg_pos = 0;
    uint8_t *midi_tmp;

    while(arg_pos < nargs)
    {
        switch(*arg_str++) {
            case 'h':
            case 't':
                args[arg_pos++].h = va_arg(ap->a, int64_t);
                break;
            case 'd':
                args[arg_pos++].d = va_arg(ap->a, double);
                break;
            case 'c':
            case 'i':
            case 'r':
                args[arg_pos++].i = va_arg(ap->a, int);
                break;
            case 'm':
                midi_tmp = va_arg(ap->a, uint8_t *);
                args[arg_pos].m[0] = midi_tmp[0];
                args[arg_pos].m[1] = midi_tmp[1];
                args[arg_pos].m[2] = midi_tmp[2];
                args[arg_pos++].m[3] = midi_tmp[3];
                break;
            case 'S':
            case 's':
                args[arg_pos++].s = va_arg(ap->a, const char *);
                break;
            case 'b':
                args[arg_pos].b.len = va_arg(ap->a, int);
                args[arg_pos].b.data = va_arg(ap->a, unsigned char *);
                arg_pos++;
                break;
            case 'f':
                args[arg_pos++].f = va_arg(ap->a, double);
                break;
            case 'T':
            case 'F':
            case 'N':
            case 'I':
                args[arg_pos++].T = arg_str[-1];
                break;
            default:
                ;
        }
    }
}

void rtosc_2args(rtosc_arg_t* args, size_t nargs, const char* arg_str, ...)
{
    rtosc_va_list_t va;
    va_start(va.a, arg_str);
    rtosc_v2args(args, nargs, arg_str, &va);
    va_end(va.a);
}

void rtosc_v2argvals(rtosc_arg_val_t* args, size_t nargs, const char* arg_str, va_list ap)
{
    rtosc_va_list_t ap2;
    va_copy(ap2.a, ap);
    for(size_t i=0; i<nargs; ++i, ++arg_str, ++args)
    {
        args->type = *arg_str;
        rtosc_v2args(&args->val, 1, arg_str, &ap2);
    }
}

void rtosc_2argvals(rtosc_arg_val_t* args, size_t nargs, const char* arg_str, ...)
{
    va_list va;
    va_start(va, arg_str);
    rtosc_v2argvals(args, nargs, arg_str, va);
    va_end(va);
}

size_t rtosc_vmessage(char   *buffer,
                      size_t      len,
                      const char *address,
                      const char *arguments,
                      va_list ap)
{
    const unsigned nargs = nreserved(arguments);
    if(!nargs)
        return rtosc_amessage(buffer,len,address,arguments,NULL);

    rtosc_arg_t args[nargs];
    rtosc_va_list_t ap2;
    va_copy(ap2.a, ap);
    rtosc_v2args(args, nargs, arguments, &ap2);

    return rtosc_amessage(buffer,len,address,arguments,args);
}

size_t rtosc_amessage(char              *buffer,
                      size_t             len,
                      const char        *address,
                      const char        *arguments,
                      const rtosc_arg_t *args)
{
    const size_t total_len = vsosc_null(address, arguments, args);

    if(!buffer)
        return total_len;

    //Abort if the message cannot fit
    if(total_len>len) {
        memset(buffer, 0, len);
        return 0;
    }

    memset(buffer, 0, total_len);

    unsigned pos = 0;
    while(*address)
        buffer[pos++] = *address++;

    //get 32 bit alignment
    pos += 4-pos%4;

    buffer[pos++] = ',';

    const char *arg_str = arguments;
    while(*arg_str)
        buffer[pos++] = *arg_str++;

    pos += 4-pos%4;

    unsigned toparse = nreserved(arguments);
    unsigned arg_pos = 0;
    while(toparse)
    {
        char arg = *arguments++;
        assert(arg);
        int32_t i;
        int64_t d;
        const uint8_t *m;
        const char *s;
        const unsigned char *u;
        rtosc_blob_t b;
        switch(arg) {
            case 'h':
            case 't':
            case 'd':
                d = args[arg_pos++].t;
                buffer[pos++] = ((d>>56) & 0xff);
                buffer[pos++] = ((d>>48) & 0xff);
                buffer[pos++] = ((d>>40) & 0xff);
                buffer[pos++] = ((d>>32) & 0xff);
                buffer[pos++] = ((d>>24) & 0xff);
                buffer[pos++] = ((d>>16) & 0xff);
                buffer[pos++] = ((d>>8) & 0xff);
                buffer[pos++] = (d & 0xff);
                --toparse;
                break;
            case 'r':
            case 'f':
            case 'c':
            case 'i':
                i = args[arg_pos++].i;
                buffer[pos++] = ((i>>24) & 0xff);
                buffer[pos++] = ((i>>16) & 0xff);
                buffer[pos++] = ((i>>8) & 0xff);
                buffer[pos++] = (i & 0xff);
                --toparse;
                break;
            case 'm':
                //TODO verify ordering of spec
                m = args[arg_pos++].m;
                buffer[pos++] = m[0];
                buffer[pos++] = m[1];
                buffer[pos++] = m[2];
                buffer[pos++] = m[3];
                --toparse;
                break;
            case 'S':
            case 's':
                s = args[arg_pos++].s;
                while(*s)
                    buffer[pos++] = *s++;
                pos += 4-pos%4;
                --toparse;
                break;
            case 'b':
                b = args[arg_pos++].b;
                i = b.len;
                buffer[pos++] = ((i>>24) & 0xff);
                buffer[pos++] = ((i>>16) & 0xff);
                buffer[pos++] = ((i>>8) & 0xff);
                buffer[pos++] = (i & 0xff);
                u = b.data;
                if(u) {
                    while(i--)
                        buffer[pos++] = *u++;
                }
                else
                    pos += i;
                if(pos%4)
                    pos += 4-pos%4;
                --toparse;
                break;
            default:
                ;
        }
    }

    return pos;
}

static rtosc_arg_t extract_arg(const uint8_t *arg_pos, char type)
{
    rtosc_arg_t result = {0};
    //trivial case
    if(!has_reserved(type)) {
        switch(type)
        {
            case 'T':
                result.T = true;
                break;
            case 'F':
                result.T = false;
                break;
            default:
                ;
        }
    } else {
        switch(type)
        {
            case 'h':
            case 't':
            case 'd':
                result.t |= (((uint64_t)*arg_pos++) << 56);
                result.t |= (((uint64_t)*arg_pos++) << 48);
                result.t |= (((uint64_t)*arg_pos++) << 40);
                result.t |= (((uint64_t)*arg_pos++) << 32);
                result.t |= (((uint64_t)*arg_pos++) << 24);
                result.t |= (((uint64_t)*arg_pos++) << 16);
                result.t |= (((uint64_t)*arg_pos++) << 8);
                result.t |= (((uint64_t)*arg_pos++));
                break;
            case 'r':
            case 'f':
            case 'c':
            case 'i':
                result.i |= (*arg_pos++ << 24);
                result.i |= (*arg_pos++ << 16);
                result.i |= (*arg_pos++ << 8);
                result.i |= (*arg_pos++);
                break;
            case 'm':
                result.m[0] = *arg_pos++;
                result.m[1] = *arg_pos++;
                result.m[2] = *arg_pos++;
                result.m[3] = *arg_pos++;
                break;
            case 'b':
                result.b.len |= (*arg_pos++ << 24);
                result.b.len |= (*arg_pos++ << 16);
                result.b.len |= (*arg_pos++ << 8);
                result.b.len |= (*arg_pos++);
                result.b.data = (unsigned char *)arg_pos;
                break;
            case 'S':
            case 's':
                result.s = (char *)arg_pos;
                break;
        }
    }

    return result;
}

static const char *advance_past_dummy_args(const char *args)
{
    while(*args == '[' || *args == ']')
        args++;
    return args;
}

rtosc_arg_itr_t rtosc_itr_begin(const char *msg)
{
    rtosc_arg_itr_t itr;
    itr.type_pos  = advance_past_dummy_args(rtosc_argument_string(msg));
    itr.value_pos = (uint8_t*)(msg+arg_start(msg));

    return itr;
}

rtosc_arg_val_t rtosc_itr_next(rtosc_arg_itr_t *itr)
{
    //current position provides the value
    rtosc_arg_val_t result = {0,{0}};
    result.type = *itr->type_pos;
    if(result.type)
        result.val = extract_arg(itr->value_pos, result.type);

    //advance
    itr->type_pos = advance_past_dummy_args(itr->type_pos+1);
    char type = result.type;
    int size  = arg_size(itr->value_pos, type);
    itr->value_pos += size;


    return result;
}

int rtosc_itr_end(rtosc_arg_itr_t itr)
{
    return !itr.type_pos  || !*itr.type_pos;
}

rtosc_arg_t rtosc_argument(const char *msg, unsigned idx)
{
    char type = rtosc_type(msg, idx);
    uint8_t *arg_mem = (uint8_t*)msg + arg_off(msg, idx);
    return extract_arg(arg_mem, type);
}

static unsigned char deref(unsigned pos, ring_t *ring)
{
    return pos<ring[0].len ? ring[0].data[pos] :
        ((pos-ring[0].len)<ring[1].len ? ring[1].data[pos-ring[0].len] : 0x00);
}

static size_t bundle_ring_length(ring_t *ring)
{
    unsigned pos = 8+8;//goto first length field
    uint32_t advance = 0;
    do {
        advance = deref(pos+0, ring) << (8*3) |
                  deref(pos+1, ring) << (8*2) |
                  deref(pos+2, ring) << (8*1) |
                  deref(pos+3, ring) << (8*0);
        if(advance)
            pos += 4+advance;
    } while(advance);

    return pos <= (ring[0].len+ring[1].len) ? pos : 0;
}

//Zero means no full message present
size_t rtosc_message_ring_length(ring_t *ring)
{
    //Check if the message is a bundle
    if(deref(0,ring) == '#' &&
            deref(1,ring) == 'b' &&
            deref(2,ring) == 'u' &&
            deref(3,ring) == 'n' &&
            deref(4,ring) == 'd' &&
            deref(5,ring) == 'l' &&
            deref(6,ring) == 'e' &&
            deref(7,ring) == '\0')
        return bundle_ring_length(ring);

    //Proceed for normal messages
    //Consume path
    unsigned pos = 0;
    while(deref(pos++,ring));
    pos--;

    //Travel through the null word end [1..4] bytes
    for(int i=0; i<4; ++i)
        if(deref(++pos, ring))
            break;

    if(deref(pos, ring) != ',')
        return 0;

    unsigned aligned_pos = pos;
    int arguments = pos+1;
    while(deref(++pos,ring));
    pos += 4-(pos-aligned_pos)%4;

    unsigned toparse = 0;
    {
        int arg = arguments-1;
        while(deref(++arg,ring))
            toparse += has_reserved(deref(arg,ring));
    }

    //Take care of varargs
    while(toparse)
    {
        char arg = deref(arguments++,ring);
        assert(arg);
        uint32_t i;
        switch(arg) {
            case 'h':
            case 't':
            case 'd':
                pos += 8;
                --toparse;
                break;
            case 'm':
            case 'r':
            case 'c':
            case 'f':
            case 'i':
                pos += 4;
                --toparse;
                break;
            case 'S':
            case 's':
                while(deref(++pos,ring));
                pos += 4-(pos-aligned_pos)%4;
                --toparse;
                break;
            case 'b':
                i = 0;
                i |= (deref(pos++,ring) << 24);
                i |= (deref(pos++,ring) << 16);
                i |= (deref(pos++,ring) << 8);
                i |= (deref(pos++,ring));
                pos += i;
                if((pos-aligned_pos)%4)
                    pos += 4-(pos-aligned_pos)%4;
                --toparse;
                break;
            default:
                ;
        }
    }


    return pos <= (ring[0].len+ring[1].len) ? pos : 0;
}

size_t rtosc_message_length(const char *msg, size_t len)
{
    ring_t ring[2] = {{(char*)msg,len},{NULL,0}};
    return rtosc_message_ring_length(ring);
}

bool rtosc_valid_message_p(const char *msg, size_t len)
{
    //Validate Path Characters (assumes printable characters are sufficient)
    if(*msg != '/')
        return false;
    const char *tmp = msg;
    for(unsigned i=0; i<len; ++i) {
        if(*tmp == 0)
            break;
        if(!isprint(*tmp))
            return false;
        tmp++;
    }

    //tmp is now either pointing to a null or the end of the string
    const size_t offset1 = tmp-msg;
    size_t       offset2 = tmp-msg;
    for(; offset2<len; offset2++) {
        if(*tmp == ',')
            break;
        tmp++;
    }

    //Too many NULL bytes
    if(offset2-offset1 > 4)
        return false;

    if((offset2 % 4) != 0)
        return false;

    size_t observed_length = rtosc_message_length(msg, len);
    return observed_length == len;
}
static uint64_t extract_uint64(const uint8_t *arg_pos)
{
    uint64_t arg = 0;
    arg |= (((uint64_t)*arg_pos++) << 56);
    arg |= (((uint64_t)*arg_pos++) << 48);
    arg |= (((uint64_t)*arg_pos++) << 40);
    arg |= (((uint64_t)*arg_pos++) << 32);
    arg |= (((uint64_t)*arg_pos++) << 24);
    arg |= (((uint64_t)*arg_pos++) << 16);
    arg |= (((uint64_t)*arg_pos++) << 8);
    arg |= (((uint64_t)*arg_pos++));
    return arg;
}

static uint32_t extract_uint32(const uint8_t *arg_pos)
{
    uint32_t arg = 0;
    arg |= (((uint32_t)*arg_pos++) << 24);
    arg |= (((uint32_t)*arg_pos++) << 16);
    arg |= (((uint32_t)*arg_pos++) << 8);
    arg |= (((uint32_t)*arg_pos++));
    return arg;
}

static void emplace_uint64(uint8_t *buffer, uint64_t d)
{
    buffer[0] = ((d>>56) & 0xff);
    buffer[1] = ((d>>48) & 0xff);
    buffer[2] = ((d>>40) & 0xff);
    buffer[3] = ((d>>32) & 0xff);
    buffer[4] = ((d>>24) & 0xff);
    buffer[5] = ((d>>16) & 0xff);
    buffer[6] = ((d>>8)  & 0xff);
    buffer[7] = ((d>>0)  & 0xff);
}

static void emplace_uint32(uint8_t *buffer, uint32_t d)
{
    buffer[0] = ((d>>24) & 0xff);
    buffer[1] = ((d>>16) & 0xff);
    buffer[2] = ((d>>8)  & 0xff);
    buffer[3] = ((d>>0)  & 0xff);
}

size_t rtosc_bundle(char *buffer, size_t len, uint64_t tt, int elms, ...)
{
    char *_buffer = buffer;
    memset(buffer, 0, len);
    strcpy(buffer, "#bundle");
    buffer += 8;
    emplace_uint64((uint8_t*)buffer, tt);
    buffer += 8;
    va_list va;
    va_start(va, elms);
    for(int i=0; i<elms; ++i) {
        const char   *msg  = va_arg(va, const char*);
        //It is assumed that any passed message/bundle is valid
        size_t        size = rtosc_message_length(msg, -1);
        emplace_uint32((uint8_t*)buffer, size);
        buffer += 4;
        memcpy(buffer, msg, size);
        buffer+=size;
    }
    va_end(va);

    return buffer-_buffer;
}


#define POS ((size_t)(((const char *)lengths) - buffer))
size_t rtosc_bundle_elements(const char *buffer, size_t len)
{
    const uint32_t *lengths = (const uint32_t*) (buffer+16);
    size_t elms = 0;
    while(POS < len && extract_uint32((const uint8_t*)lengths)) {
        lengths += extract_uint32((const uint8_t*)lengths)/4+1;

        if(POS > len)
            break;
        ++elms;
    }
    return elms;
}
#undef POS

const char *rtosc_bundle_fetch(const char *buffer, unsigned elm)
{
    const uint32_t *lengths = (const uint32_t*) (buffer+16);
    size_t elm_pos = 0;
    while(elm_pos!=elm && extract_uint32((const uint8_t*)lengths)) {
        ++elm_pos;
        lengths += extract_uint32((const uint8_t*)lengths)/4+1;
    }

    return (const char*) (elm==elm_pos?lengths+1:NULL);
}

size_t rtosc_bundle_size(const char *buffer, unsigned elm)
{
    const uint32_t *lengths = (const uint32_t*) (buffer+16);
    size_t elm_pos = 0;
    size_t last_len = 0;
    while(elm_pos!=elm && extract_uint32((const uint8_t*)lengths)) {
        last_len = extract_uint32((const uint8_t*)lengths);
        ++elm_pos, lengths+=extract_uint32((const uint8_t*)lengths)/4+1;
    }

    return last_len;
}

int rtosc_bundle_p(const char *msg)
{
    return !strcmp(msg,"#bundle");
}

uint64_t rtosc_bundle_timetag(const char *msg)
{
    return extract_uint64((const uint8_t*)msg+8);
}
